using System;
using System.IO;
using DynaPDF;

namespace text_formatting
{
	class TextFormatting
	{
      /*
         Note that the dynapdf.dll must be copied into the output directory or into a
         Windwos search path (e.g. %WINDOWS%/System32) before the application can be executed!
      */

      // Error callback function.
      // If the function name should not appear at the beginning of the error message then set
      // the flag emNoFuncNames (pdf.SetErrorMode(DynaPDF.TErrMode.emNoFuncNames);).
      static int PDFError(IntPtr Data, int ErrCode, IntPtr ErrMessage, int ErrType)
      {
         Console.Write("{0}\n", System.Runtime.InteropServices.Marshal.PtrToStringAnsi(ErrMessage));
         return 0; // We try to continue if an error occurrs. Any other return value breaks processing.
      }

      private struct TOutRect
      {
         public CPDF   pdf;      // Active PDF instance
         public double PosX;     // Original x-coordinate of first output rectangle
         public double PosY;     // Original y-coordinate of first output rectangle
         public double Width;    // Original width of first output rectangle
         public double Height;   // Original height of first output rectangle
         public double Distance; // Space between columns
         public int    Column;   // Current column
         public int    ColCount; // Number of colummns
      };

      static private TOutRect m_OutRect;

      static int PageBreakProc(IntPtr Data, double LastPosX, double LastPosY, int PageBreak)
      {
         m_OutRect.pdf.SetPageCoords(TPageCoord.pcTopDown); // we use top down coordinates
         ++m_OutRect.Column;
         // PageBreak is 1 if the string contains a page break tag (see help file for further information).
         if (PageBreak == 0 && m_OutRect.Column < m_OutRect.ColCount)
         {
            // Calulate the x-coordinate of the column
            double posX = m_OutRect.PosX + m_OutRect.Column * (m_OutRect.Width + m_OutRect.Distance);
            // change the output rectangle, do not close the page!
            m_OutRect.pdf.SetTextRect(posX, m_OutRect.PosY, m_OutRect.Width, m_OutRect.Height);
            return 0; // we do not change the alignment
         }else
         {  // the page is full, close the current one and append a new page
            m_OutRect.pdf.EndPage();
            m_OutRect.pdf.Append();
            m_OutRect.pdf.SetTextRect(m_OutRect.PosX, m_OutRect.PosY, m_OutRect.Width, m_OutRect.Height);
            m_OutRect.Column = 0;
            return 0;
         }
      }

		[STAThread]
		static void Main(string[] args)
		{
		   try
		   {
            CPDF pdf = new CPDF();
            // Error messages and warnings are passed to the callback function.
            pdf.SetOnErrorProc(IntPtr.Zero, new TErrorProc(PDFError));
            pdf.CreateNewPDF(null); // We open the output file later if no error occurrs.

            pdf.SetDocInfo(TDocumentInfo.diCreator, "C# test application");
            pdf.SetDocInfo(TDocumentInfo.diTitle, "Multi-column text");

            System.IO.FileStream f = new System.IO.FileStream("../../../test_files/sample.txt", System.IO.FileMode.Open, System.IO.FileAccess.Read);
            System.IO.BinaryReader b = new System.IO.BinaryReader(f, System.Text.Encoding.Default);
            char[] buffer = new char[f.Length];
            b.Read(buffer, 0, (int)f.Length);
            f.Close();

            pdf.SetPageCoords(TPageCoord.pcTopDown);
            pdf.Append();

            // We us a private data type to store the properties of the output retangle.
            // The structure is required to calculate the new string position within the
            // callback function. In other programming languages we would pass this structure as a pointer
            // to the callback function. However, such a concept does not work with .Net languages...
            m_OutRect.pdf      = pdf;
            m_OutRect.ColCount = 3;    // Set the number of columns
            m_OutRect.Column   = 0;    // Current column
            m_OutRect.Distance = 10.0; // Distance between two columns
            m_OutRect.PosX     = 50.0;
            m_OutRect.PosY     = 50.0;
            m_OutRect.Height   = pdf.GetPageHeight() - m_OutRect.PosY * 2.0;
            m_OutRect.Width    = (pdf.GetPageWidth() - m_OutRect.PosX * 2.0 - (m_OutRect.ColCount -1) * m_OutRect.Distance) / m_OutRect.ColCount;

            // The callback function is called if either the page is full or if a page break operator is found.
            pdf.SetOnPageBreakProc(IntPtr.Zero, new TOnPageBreakProc(PageBreakProc));
            // Set the output rectangle first
            pdf.SetTextRect(m_OutRect.PosX, m_OutRect.PosY, m_OutRect.Width, m_OutRect.Height);

            pdf.SetFont("Arial", TFStyle.fsNone, 10.0, true, TCodepage.cp1252);
            pdf.WriteFText(TTextAlign.taJustify, new String(buffer)); // Now we can print the text
            pdf.EndPage();

            // No fatal error occurred?
            if (pdf.HaveOpenDoc())
            {
               // We write the output file into the current directory.
               String filePath = Directory.GetCurrentDirectory() + "\\out.pdf";
               // OK, now we can open the output file.
               if (!pdf.OpenOutputFile(filePath)) return;
               if (pdf.CloseFile())
               {
                  Console.Write("PDF file \"{0}\" successfully created!\n", filePath);
                  System.Diagnostics.Process p = new System.Diagnostics.Process();
                  p.StartInfo.FileName = filePath;
                  p.Start();
               }
            }
            pdf = null;
         }catch(Exception e)
         {
            Console.Write(e.Message);
         }
         Console.Read();
		}
	}
}
